<?php
namespace Smartwave\PortoChild\Model\Layer\Filter;

use Magento\Catalog\Model\Layer;
use Magento\Framework\Exception\StateException;
use Magento\Search\Model\SearchEngine;
use Amasty\Shopby\Helper\FilterSetting;
use Magento\Store\Model\StoreManagerInterface;
use Amasty\ShopbyBase\Api\Data\FilterSettingInterface;
use Magento\Catalog\Model\Layer\Filter\AbstractFilter;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Amasty\Shopby\Model\Layer\Filter\Traits\FilterTrait;
use Amasty\Shopby\Helper\Group as GroupHelper;
use Magento\Framework\Filter\StripTags as TagFilter;
use Magento\Catalog\Model\Layer\Filter\ItemFactory as FilterItemFactory;
use Magento\Catalog\Model\Layer\Filter\Item\DataBuilder as ItemDataBuilder;
use Amasty\Shopby\Model\Search\RequestGenerator as ShopbyRequestGenerator;
use Amasty\Shopby\Model\Search\Adapter\Mysql\AggregationAdapter as MysqlAggregationAdapter;
use \Magento\Store\Model\Store;

/**
 * Layer attribute filter
 */
class Inventory extends AbstractFilter
{
    use FilterTrait;

    /**
     * @var TagFilter
     */
    private $tagFilter;

    /**
     * @var FilterSettingInterface
     */
    private $filterSetting;

    /**
     * @var MysqlAggregationAdapter
     */
    private $aggregationAdapter;

    /**
     * @var SearchEngine
     */
    private $searchEngine;

    /**
     * @var  FilterSetting
     */
    private $settingHelper;

    /**
     * @var  ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var \Amasty\Shopby\Model\Request
     */
    private $shopbyRequest;

    /**
     * @var GroupHelper
     */
    private $groupHelper;

    /**
     * @var \Amasty\ShopbyBase\Helper\OptionSetting
     */
    private $optionSettingHelper;

    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;

    private $inventoryHelper;
    private $productRepository;

    public function __construct(
        FilterItemFactory $filterItemFactory,
        StoreManagerInterface $storeManager,
        Layer $layer,
        ItemDataBuilder $itemDataBuilder,
        TagFilter $tagFilter,
        MysqlAggregationAdapter $aggregationAdapter,
        SearchEngine $searchEngine,
        FilterSetting $settingHelper,
        ScopeConfigInterface $scopeConfig,
        \Amasty\Shopby\Model\Request $shopbyRequest,
        GroupHelper $groupHelper,
        \Amasty\ShopbyBase\Helper\OptionSetting $optionSettingHelper,
        \Magento\Framework\Message\ManagerInterface $messageManager,
        \Smartwave\PortoChild\Helper\Warehouse $inventoryHelper,
        \Magento\Catalog\Model\ProductRepository $productRepository,
        array $data = []
    ) {
        parent::__construct(
            $filterItemFactory,
            $storeManager,
            $layer,
            $itemDataBuilder,
            $data
        );

        $this->tagFilter = $tagFilter;
        $this->settingHelper = $settingHelper;
        $this->aggregationAdapter = $aggregationAdapter;
        $this->shopbyRequest = $shopbyRequest;
        $this->groupHelper = $groupHelper;
        $this->scopeConfig = $scopeConfig;
        $this->searchEngine = $searchEngine;
        $this->optionSettingHelper = $optionSettingHelper;
        $this->messageManager = $messageManager;
        $this->inventoryHelper = $inventoryHelper;
        $this->productRepository = $productRepository;
    }

    /**
     * Apply attribute option filter to product collection.
     *
     * @param \Magento\Framework\App\RequestInterface $request
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */

    public function apply(\Magento\Framework\App\RequestInterface $request)
    {
        $logger = \Magento\Framework\App\ObjectManager::getInstance()->get('Psr\Log\LoggerInterface');
        if ($this->isApplied()) {
            return $this;
        }

        $requestedOptionsString = $this->shopbyRequest->getFilterParam($this);

        if (empty($requestedOptionsString)) {
            return $this;
        }

        $requestedOptions = explode(',', $requestedOptionsString);

        $this->setCurrentValue($requestedOptions);
        $this->addState($requestedOptions);

        if (!$this->isMultiSelectAllowed() && count($requestedOptions) > 1) {
            $requestedOptions = array_slice($requestedOptions, 0, 1);
        }

        $attribute = $this->getAttributeModel();

        /** @var \Amasty\Shopby\Model\ResourceModel\Fulltext\Collection $productCollection */
        $productCollection = $this->getLayer()->getProductCollection();
        $ids = $productCollection->getAllIds();

        $filter_sku = array();
        foreach ($ids as $id) {
            $p = $this->productRepository->getById($id);
            $sku = $p->getSku();
            $logger->debug($p->getName().':'.$sku);

            if ($this->hasInventorySource($sku, $requestedOptions)) {
                $filter_sku[] = $sku;
                $logger->debug('>filter:'.$sku);
            }
        }
        $productCollection->addAttributeToFilter( 'sku', array('in'=>$filter_sku) );

        return $this;
    }

    protected function hasInventorySource($sku, $ivs) {
        $logger = \Magento\Framework\App\ObjectManager::getInstance()->get('Psr\Log\LoggerInterface');
        $_sources = $this->inventoryHelper->getInventoriesBySource($sku);
        foreach ($_sources as $_item) {
            $logger->debug($_item['code']);
            if (in_array($_item['code'], $ivs, true)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get data array for building attribute filter items.
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function _getItemsData()
    {
        $selected = !!$this->shopbyRequest->getFilterParam($this);
        if ($selected && !$this->isVisibleWhenSelected()) {
            return [];
        }

        // $logger = \Magento\Framework\App\ObjectManager::getInstance()->get('Psr\Log\LoggerInterface');
        // $logger->debug(print_r($options, true));
        $options = $this->getOptions();
        
        $optionsFacetedData = $this->getOptionsFacetedData($options);
        if (!$optionsFacetedData) {
            return [];
        }

        $this->addItemsToDataBuilder($options, $optionsFacetedData);

        $itemsData = $this->getItemsFromDataBuilder();

        return $itemsData;
    }

    /**
     * @param array $values
     */
    private function addState(array $values)
    {
        if (!$this->shouldAddState()) {
            return;
        }

        $labels = [];

        $attribute = $this->getAttributeModel();

        foreach ($values as $value) {
            $labelGroup = null;
            if ($attributeId = $attribute->getAttributeId()) {
                $labelGroup = $this->groupHelper->getGroupLabel($attributeId, $value);
            }
            if ($labelGroup) {
                $labels[] = $labelGroup;
            } else {
                // $labels[] = $this->getOptionText($value);
                $labels[] = $this->inventoryHelper->getInventorySourceLabel($value);
            }
        }

        foreach ($labels as $label) {
            $item = $this->_createItem($label, $values);
            $this->getLayer()->getState()
                ->addFilter(
                    $item
                );
        }
    }

    /**
     * @return bool
     */
    public function shouldAddState()
    {
        // Could be overwritten in plugins.
        return true;
    }

    /**
     * @return bool
     */
    private function isMultiSelectAllowed()
    {
        return $this->getFilterSetting()->isMultiselect();
    }

    /**
     * @return FilterSettingInterface
     */
    protected function getFilterSetting()
    {
        if ($this->filterSetting === null) {
            $this->filterSetting = $this->settingHelper->getSettingByLayerFilter($this);
        }
        return $this->filterSetting;
    }

    /**
     * @param string $attributeCode
     * @param $key
     * @return string
     */
    private function getFakeAttributeCodeForApply($attributeCode, $key)
    {
        if ($key > 0) {
            $attributeCode .= ShopbyRequestGenerator::FAKE_SUFFIX . $key;
        }

        return $attributeCode;
    }

    /**
     * @return int
     */
    public function getItemsCount()
    {
        return count($this->getItems());
    }

    /**
     * @param array $a
     * @param array $b
     * @return int
     */
    public function sortOption($a, $b)
    {
        $pattern = '@^(\d+)@';
        if (preg_match($pattern, $a['label'], $ma) && preg_match($pattern, $b['label'], $mb)) {
            $r = $ma[1] - $mb[1];
            if ($r != 0) {
                return $r;
            }
        }

        return strcasecmp($a['label'], $b['label']);
    }

    /**
     * @return array
     */
    private function getOptions()
    {
        $objectManager =  \Magento\Framework\App\ObjectManager::getInstance();
        $options = $this->inventoryHelper->getInventorySourceList();

        return $options;
    }

    private function getOptionsFacetedData($options)
    {
        $facetedData = array();
        foreach ($options as $o) {
            if (!empty($o['value'])) {
                $facetedData[ $o['value'] ] = array(
                    'value' => $o['value'],
                    'count' => 1
                );
            }
        }
        return $facetedData;
    }

    /**
     * @return array
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function getOptionsFacetedData2()
    {
        $optionsFacetedData = $this->generateOptionsFacetedData();
        $optionsFacetedData = $this->adjustFacetedDataToGroup($optionsFacetedData);

        if (count($optionsFacetedData)) {
            $optionsFacetedData = $this->convertOptionsFacetedData($optionsFacetedData);
        }

        return $optionsFacetedData;
    }

    /**
     * @param array $optionsFacetedData
     *
     * @return array
     */
    protected function convertOptionsFacetedData($optionsFacetedData)
    {
        $attributeValue = $this->shopbyRequest->getFilterParam($this);
        $values = explode(",", $attributeValue);
        foreach ($values as $value) {
            if (!empty($value) && !array_key_exists($value, $optionsFacetedData)) {
                $optionsFacetedData[$value] = ['value' => $value, 'count' => 0];
            }
        }

        return $optionsFacetedData;
    }

    /**
     * @return array
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function generateOptionsFacetedData()
    {
        /** @var \Amasty\Shopby\Model\ResourceModel\Fulltext\Collection $productCollectionOrigin */
        $productCollectionOrigin = $this->getLayer()->getProductCollection();
        $attribute = $this->getAttributeModel();

        try {
            $optionsFacetedData = $productCollectionOrigin->getFacetedData(
                $attribute->getAttributeCode(),
                $this->getAlteredQueryResponse()
            );
        } catch (StateException $e) {
            if (!$this->messageManager->hasMessages()) {
                $this->messageManager->addErrorMessage(
                    __(
                        'Make sure that "%1" attribute can be used in layered navigation',
                        $attribute->getAttributeCode()
                    )
                );
            }
            $optionsFacetedData = [];
        }

        return $optionsFacetedData;
    }

    /**
     * @return \Magento\Framework\Search\ResponseInterface|null
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function getAlteredQueryResponse()
    {
        $alteredQueryResponse = null;

        if ($this->hasCurrentValue() && !$this->getFilterSetting()->isUseAndLogic()) {
            $requestBuilder = $this->getRequestBuilder();
            $queryRequest = $requestBuilder->create();
            $alteredQueryResponse = $this->searchEngine->search($queryRequest);
        }

        return $alteredQueryResponse;
    }

    /**
     * @return \Amasty\Shopby\Model\Request\Builder
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function getRequestBuilder()
    {
        $requestBuilder = $this->getMemRequestBuilder();
        $attributeCode = $this->getAttributeModel()->getAttributeCode();
        $requestBuilder->removePlaceholder($attributeCode);
        $requestBuilder->setAggregationsOnly($attributeCode);

        return $requestBuilder;
    }

    /**
     * @param array $optionsFacetedData
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function adjustFacetedDataToGroup(array $optionsFacetedData)
    {
        if (!$optionsFacetedData) {
            return $optionsFacetedData;
        }

        $groups = $this->groupHelper->getGroupsByAttributeId($this->getAttributeModel()->getId());

        foreach ($groups as $group) {
            $key = GroupHelper::LAST_POSSIBLE_OPTION_ID - $group->getId();

            if (isset($optionsFacetedData[$key])) {
                $code = $group->getGroupCode();
                $optionsFacetedData[$code] = $optionsFacetedData[$key];
                unset($optionsFacetedData[$key]);
            }
        }

        return $optionsFacetedData;
    }

    /**
     * @param array $options
     * @param array $optionsFacetedData
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function addItemsToDataBuilder($options, $optionsFacetedData)
    {
        if (!$options) {
            return;
        }
        foreach ($options as $option) {
            if (empty($option['value'])) {
                continue;
            }

            $isFilterableAttribute = $this->getAttributeIsFilterable($this->getAttributeModel());
            if (isset($optionsFacetedData[$option['value']])
                || $isFilterableAttribute != static::ATTRIBUTE_OPTIONS_ONLY_WITH_RESULTS
            ) {
                $count = 0;
                if (isset($optionsFacetedData[$option['value']]['count'])) {
                    $count = $optionsFacetedData[$option['value']]['count'];
                }
                $this->itemDataBuilder->addItemData(
                    $this->tagFilter->filter($option['label']),
                    $option['value'],
                    $count
                );
            }
        }
    }

    /**
     * Get items data according to attribute settings.
     * @return array
     */
    private function getItemsFromDataBuilder()
    {
        $itemsData = $this->itemDataBuilder->build();
        if (count($itemsData) == 1
            && !$this->isOptionReducesResults(
                $itemsData[0]['count'],
                $this->getLayer()->getProductCollection()->getSize()
            )
        ) {
            $itemsData = $this->getReducedItemsData($itemsData);
        }

        return $itemsData;
    }
}
